{
  title: 'Egnyte',

  connection: {
    fields: [
      {
        name: 'environment',
        control_type: 'select',
        pick_list: [%w[Production egnyte.com], %w[Sandbox qa-egnyte.com]],
        optional: false
      },
      {
        ngIf: 'input.environment == "egnyte.com"',
        name: 'domain',
        label: 'Egnyte domain',
        hint: "This is the user's Egnyte domain. An example domain name " \
          "is 'acme', if instance URL is https://acme.egnyte.com. Note that " \
          'initially your API key will only work with the single Egnyte ' \
          'domain that you specified, you would use for testing.',
        control_type: 'subdomain',
        url: '.egnyte.com',
        optional: false
      },
      {
        ngIf: 'input.environment == "qa-egnyte.com"',
        name: 'domain',
        label: 'Egnyte domain',
        hint: "This is the user's Egnyte domain. An example domain name " \
          "is 'acme', if instance URL is https://acme.qa-egnyte.com. Note " \
          'that initially your API key will only work with the single Egnyte ' \
          'domain that you specified, you would use for testing.',
        control_type: 'subdomain',
        url: '.qa-egnyte.com',
        optional: false
      },
      {
        name: 'client_id',
        label: 'Key',
        optional: false
      },
      {
        name: 'client_secret',
        label: 'Secret',
        optional: false,
        control_type: 'password'
      }
    ],

    authorization: {
      type: 'oauth2',

      authorization_url: lambda do |connection|
        connection_scope = 'Egnyte.filesystem'

        "https://#{connection['domain']}.#{connection['environment']}" \
          "/puboauth/token?client_id=#{connection['client_id']}" \
          '&redirect_uri=https://www.workato.com/oauth/callback' \
          "&scope=#{connection_scope}&response_type=code"
      end,

      acquire: lambda do |connection, auth_code, redirect_uri|
        # redirect_uri: redirect_uri,

        response =
          post("https://#{connection['domain']}.#{connection['environment']}" \
            '/puboauth/token',
               client_id: connection['client_id'],
               client_secret: connection['client_secret'],
               redirect_uri: redirect_uri,
               code: auth_code,
               grant_type: 'authorization_code')
          .request_format_www_form_urlencoded

        [response, nil, nil]
      end,

      refresh_on: [401, 403],

      # TODO: fix this
      # refresh: lambda do |connection, refresh_token|
      #   post("https://#{connection['environment']}.procore.com/oauth/token",
      #        grant_type: 'refresh_token',
      #        client_id: connection['client_id'],
      #        client_secret: connection['client_secret'],
      #        refresh_token: refresh_token,
      #        redirect_uri: 'https://www.workato.com/oauth/callback')
      #     .request_format_www_form_urlencoded
      # end,

      apply: lambda do |_connection, access_token|
        headers('Authorization' => "Bearer #{access_token}")
      end
    },

    base_uri: lambda do |connection|
      "https://#{connection['domain']}.#{connection['environment']}"
    end
  },

  test: ->(_connection) { get('/pubapi/v1/userinfo') },

  methods: {
    event_sample_output: lambda do
      {
        'id' => 52,
        'timestamp' => '2019-09-26T19:37:30.000Z',
        'actor' => 3,
        'type' => 'file_system',
        'action' => 'create',
        'data' => {
          'target_path' => '/Private/sharat/Level 1/test.csv',
          'target_id' => 'ca922aec-3c76-4513-b4b8-4e6a91176ea7',
          'target_group_id' => 'deea8c78-218a-443c-be04-34e9bb8cd452',
          'is_folder' => false
        },
        'action_source' => 'WebUI',
        'object_detail' => 'https://appworkatodev.qa-egnyte.com/pubapi/v1/fs/Private/sharat/Level%201/test.csv'
      }
    end

  },

  object_definitions: {
    custom_action_input: {
      fields: lambda do |_connection, config_fields|
        input_schema = parse_json(config_fields.dig('input', 'schema') || '[]')

        [
          {
            name: 'path',
            optional: false,
            hint: 'Base URI is <b>https://www.googleapis.com</b> - path will ' \
              'be appended to this URI. ' \
              'Use absolute URI to override this base URI.'
          },
          (
            if %w[get delete].include?(config_fields['verb'])
              {
                name: 'input',
                type: 'object',
                control_type: 'form-schema-builder',
                sticky: input_schema.blank?,
                label: 'URL parameters',
                add_field_label: 'Add URL parameter',
                properties: [
                  {
                    name: 'schema',
                    extends_schema: true,
                    sticky: input_schema.blank?
                  },
                  (
                    if input_schema.present?
                      {
                        name: 'data',
                        type: 'object',
                        properties: call('make_schema_builder_fields_sticky',
                                         input_schema)
                      }
                    end
                  )
                ].compact
              }
            else
              {
                name: 'input',
                type: 'object',
                properties: [
                  {
                    name: 'schema',
                    extends_schema: true,
                    schema_neutral: true,
                    control_type: 'schema-designer',
                    sample_data_type: 'json_input',
                    sticky: input_schema.blank?,
                    label: 'Request body parameters',
                    add_field_label: 'Add request body parameter'
                  },
                  (
                    if input_schema.present?
                      {
                        name: 'data',
                        type: 'object',
                        properties: input_schema
                          .each { |field| field[:sticky] = true }
                      }
                    end
                  )
                ].compact
              }
            end
          ),
          {
            name: 'output',
            control_type: 'schema-designer',
            sample_data_type: 'json_http',
            extends_schema: true,
            schema_neutral: true,
            sticky: true
          }
        ]
      end
    },

    custom_action_output: {
      fields: lambda do |_connection, config_fields|
        parse_json(config_fields['output'] || '[]')
      end
    },

    folder: {
      fields: lambda {
                [
                  { name: 'name' },
                  { name: 'lastModified', type: 'timestamp' },
                  { name: 'count', type: 'integer' },
                  {  name:  'offset', type: 'integer' },
                  {  name:  'path', type: 'string' },
                  {  name:  'folder_id', type: 'string' },
                  { name:  'total_count', type: 'integer' },
                  { name:  'is_folder', type: 'boolean' },
                  { name:  'public_links' },
                  { name: 'folders', type: 'array', of: 'object', properties: [
                    { name: 'name' },
                    { name: 'lastModified', type: 'timestamp' },
                    { name: 'path' },
                    { name: 'folder_id', type: 'string' },
                    { name: 'is_folder', type: 'boolean' }
                  ] }
                ]
              }
    },

    folder_2: {
      fields: lambda {
                [
                  {
                    control_type: 'text',
                    label: 'Name',
                    type: 'string',
                    name: 'name'
                  },
                  {
                    control_type: 'number',
                    label: 'Last modified',
                    parse_output: 'float_conversion',
                    type: 'number',
                    name: 'lastModified'
                  },
                  {
                    control_type: 'number',
                    label: 'Count',
                    parse_output: 'float_conversion',
                    type: 'number',
                    name: 'count'
                  },
                  {
                    control_type: 'number',
                    label: 'Offset',
                    parse_output: 'float_conversion',
                    type: 'number',
                    name: 'offset'
                  },
                  {
                    control_type: 'text',
                    label: 'Path',
                    type: 'string',
                    name: 'path'
                  },
                  {
                    control_type: 'text',
                    label: 'Folder ID',
                    type: 'string',
                    name: 'folder_id'
                  },
                  {
                    control_type: 'text',
                    label: 'Parent ID',
                    type: 'string',
                    name: 'parent_id'
                  },
                  {
                    control_type: 'number',
                    label: 'Total count',
                    parse_output: 'float_conversion',
                    type: 'number',
                    name: 'total_count'
                  },
                  {
                    control_type: 'checkbox',
                    label: 'Is folder',
                    render_input: {},
                    parse_output: {},
                    toggle_hint: 'Select from option list',
                    toggle_field: {
                      label: 'Is folder',
                      control_type: 'text',
                      toggle_hint: 'Use custom value',
                      type: 'boolean',
                      name: 'is_folder'
                    },
                    type: 'boolean',
                    name: 'is_folder'
                  },
                  {
                    control_type: 'text',
                    label: 'Public links',
                    type: 'string',
                    name: 'public_links'
                  },
                  {
                    control_type: 'checkbox',
                    label: 'Allow links',
                    render_input: {},
                    parse_output: {},
                    toggle_hint: 'Select from option list',
                    toggle_field: {
                      label: 'Allow links',
                      control_type: 'text',
                      toggle_hint: 'Use custom value',
                      type: 'boolean',
                      name: 'allow_links'
                    },
                    type: 'boolean',
                    name: 'allow_links'
                  },
                  {
                    control_type: 'checkbox',
                    label: 'Restrict move delete',
                    render_input: {},
                    parse_output: {},
                    toggle_hint: 'Select from option list',
                    toggle_field: {
                      label: 'Restrict move delete',
                      control_type: 'text',
                      toggle_hint: 'Use custom value',
                      type: 'boolean',
                      name: 'restrict_move_delete'
                    },
                    type: 'boolean',
                    name: 'restrict_move_delete'
                  },
                  {
                    name: 'folders',
                    type: 'array',
                    of: 'object',
                    label: 'Folders',
                    properties: [
                      {
                        control_type: 'text',
                        label: 'Name',
                        type: 'string',
                        name: 'name'
                      },
                      {
                        control_type: 'number',
                        label: 'Last modified',
                        parse_output: 'float_conversion',
                        type: 'number',
                        name: 'lastModified'
                      },
                      {
                        control_type: 'text',
                        label: 'Path',
                        type: 'string',
                        name: 'path'
                      },
                      {
                        control_type: 'text',
                        label: 'Folder ID',
                        type: 'string',
                        name: 'folder_id'
                      },
                      {
                        control_type: 'checkbox',
                        label: 'Is folder',
                        render_input: {},
                        parse_output: {},
                        toggle_hint: 'Select from option list',
                        toggle_field: {
                          label: 'Is folder',
                          control_type: 'text',
                          toggle_hint: 'Use custom value',
                          type: 'boolean',
                          name: 'is_folder'
                        },
                        type: 'boolean',
                        name: 'is_folder'
                      }
                    ]
                  },
                  {
                    name: 'files',
                    type: 'array',
                    of: 'object',
                    label: 'Files',
                    properties: [
                      {
                        control_type: 'text',
                        label: 'Checksum',
                        type: 'string',
                        name: 'checksum'
                      },
                      {
                        control_type: 'number',
                        label: 'Size',
                        parse_output: 'float_conversion',
                        type: 'number',
                        name: 'size'
                      },
                      {
                        control_type: 'text',
                        label: 'Path',
                        type: 'string',
                        name: 'path'
                      },
                      {
                        control_type: 'text',
                        label: 'Name',
                        type: 'string',
                        name: 'name'
                      },
                      {
                        control_type: 'checkbox',
                        label: 'Locked',
                        render_input: {},
                        parse_output: {},
                        toggle_hint: 'Select from option list',
                        toggle_field: {
                          label: 'Locked',
                          control_type: 'text',
                          toggle_hint: 'Use custom value',
                          type: 'boolean',
                          name: 'locked'
                        },
                        type: 'boolean',
                        name: 'locked'
                      },
                      {
                        control_type: 'checkbox',
                        label: 'Is folder',
                        render_input: {},
                        parse_output: {},
                        toggle_hint: 'Select from option list',
                        toggle_field: {
                          label: 'Is folder',
                          control_type: 'text',
                          toggle_hint: 'Use custom value',
                          type: 'boolean',
                          name: 'is_folder'
                        },
                        type: 'boolean',
                        name: 'is_folder'
                      },
                      {
                        control_type: 'text',
                        label: 'Entry ID',
                        type: 'string',
                        name: 'entry_id'
                      },
                      {
                        control_type: 'text',
                        label: 'Group ID',
                        type: 'string',
                        name: 'group_id'
                      },
                      {
                        control_type: 'date_time',
                        label: 'Last modified',
                        render_input: 'date_time_conversion',
                        parse_output: 'date_time_conversion',
                        type: 'date_time',
                        name: 'last_modified'
                      },
                      {
                        control_type: 'text',
                        label: 'Uploaded by',
                        type: 'string',
                        name: 'uploaded_by'
                      },
                      {
                        control_type: 'number',
                        label: 'Num versions',
                        parse_output: 'float_conversion',
                        type: 'number',
                        name: 'num_versions'
                      }
                    ]
                  }
                ]
              }
    },

    file: {
      fields: lambda do |_connection, _config_fields|
        [
          {
            control_type: 'text',
            label: 'Checksum',
            type: 'string',
            name: 'checksum'
          },
          {
            control_type: 'number',
            label: 'Size',
            parse_output: 'float_conversion',
            type: 'number',
            name: 'size'
          },
          {
            control_type: 'text',
            label: 'Path',
            type: 'string',
            name: 'path'
          },
          {
            control_type: 'text',
            label: 'Name',
            type: 'string',
            name: 'name'
          },
          {
            control_type: 'checkbox',
            label: 'Locked',
            render_input: {},
            parse_output: {},
            toggle_hint: 'Select from option list',
            toggle_field: {
              label: 'Locked',
              control_type: 'text',
              toggle_hint: 'Use custom value',
              type: 'boolean',
              name: 'locked'
            },
            type: 'boolean',
            name: 'locked'
          },
          {
            control_type: 'checkbox',
            label: 'Is folder',
            render_input: {},
            parse_output: {},
            toggle_hint: 'Select from option list',
            toggle_field: {
              label: 'Is folder',
              control_type: 'text',
              toggle_hint: 'Use custom value',
              type: 'boolean',
              name: 'is_folder'
            },
            type: 'boolean',
            name: 'is_folder'
          },
          {
            control_type: 'text',
            label: 'Entry ID',
            type: 'string',
            name: 'entry_id'
          },
          {
            control_type: 'text',
            label: 'Group ID',
            type: 'string',
            name: 'group_id'
          },
          {
            control_type: 'date_time',
            label: 'Last modified',
            render_input: 'date_time_conversion',
            parse_output: 'date_time_conversion',
            type: 'date_time',
            name: 'last_modified'
          },
          {
            control_type: 'text',
            label: 'Uploaded by',
            type: 'string',
            name: 'uploaded_by'
          },
          {
            control_type: 'number',
            label: 'Num versions',
            parse_output: 'float_conversion',
            type: 'number',
            name: 'num_versions'
          }
        ]
      end
    },

    info: {
      fields: lambda do |_connection, _config_fields|
        [
          {
            control_type: 'text',
            label: 'Name',
            type: 'string',
            name: 'name'
          },
          {
            control_type: 'number',
            label: 'Last modified',
            parse_output: 'float_conversion',
            type: 'number',
            name: 'lastModified'
          },
          {
            control_type: 'number',
            label: 'Uploaded',
            parse_output: 'float_conversion',
            type: 'number',
            name: 'uploaded'
          },
          {
            control_type: 'number',
            label: 'Count',
            parse_output: 'float_conversion',
            type: 'number',
            name: 'count'
          },
          {
            control_type: 'number',
            label: 'Offset',
            parse_output: 'float_conversion',
            type: 'number',
            name: 'offset'
          },
          {
            control_type: 'text',
            label: 'Path',
            type: 'string',
            name: 'path'
          },
          {
            control_type: 'text',
            label: 'Folder ID',
            type: 'string',
            name: 'folder_id'
          },
          {
            control_type: 'text',
            label: 'Parent ID',
            type: 'string',
            name: 'parent_id'
          },
          {
            control_type: 'number',
            label: 'Total count',
            parse_output: 'float_conversion',
            type: 'number',
            name: 'total_count'
          },
          {
            control_type: 'checkbox',
            label: 'Is folder',
            render_input: {},
            parse_output: {},
            toggle_hint: 'Select from option list',
            toggle_field: {
              label: 'Is folder',
              control_type: 'text',
              toggle_hint: 'Use custom value',
              type: 'boolean',
              name: 'is_folder'
            },
            type: 'boolean',
            name: 'is_folder'
          },
          {
            control_type: 'text',
            label: 'Public links',
            type: 'string',
            name: 'public_links'
          },
          {
            control_type: 'checkbox',
            label: 'Allow links',
            render_input: {},
            parse_output: {},
            toggle_hint: 'Select from option list',
            toggle_field: {
              label: 'Allow links',
              control_type: 'text',
              toggle_hint: 'Use custom value',
              type: 'boolean',
              name: 'allow_links'
            },
            type: 'boolean',
            name: 'allow_links'
          },
          {
            control_type: 'checkbox',
            label: 'Restrict move delete',
            render_input: {},
            parse_output: {},
            toggle_hint: 'Select from option list',
            toggle_field: {
              label: 'Restrict move delete',
              control_type: 'text',
              toggle_hint: 'Use custom value',
              type: 'boolean',
              name: 'restrict_move_delete'
            },
            type: 'boolean',
            name: 'restrict_move_delete'
          },
          {
            name: 'folders',
            type: 'array',
            of: 'object',
            label: 'Folders',
            properties: [
              {
                control_type: 'text',
                label: 'Name',
                type: 'string',
                name: 'name'
              },
              {
                control_type: 'number',
                label: 'Last modified',
                parse_output: 'float_conversion',
                type: 'number',
                name: 'lastModified'
              },
              {
                control_type: 'number',
                label: 'Uploaded',
                parse_output: 'float_conversion',
                type: 'number',
                name: 'uploaded'
              },
              {
                control_type: 'text',
                label: 'Path',
                type: 'string',
                name: 'path'
              },
              {
                control_type: 'text',
                label: 'Folder ID',
                type: 'string',
                name: 'folder_id'
              },
              {
                control_type: 'checkbox',
                label: 'Is folder',
                render_input: {},
                parse_output: {},
                toggle_hint: 'Select from option list',
                toggle_field: {
                  label: 'Is folder',
                  control_type: 'text',
                  toggle_hint: 'Use custom value',
                  type: 'boolean',
                  name: 'is_folder'
                },
                type: 'boolean',
                name: 'is_folder'
              },
              {
                control_type: 'text',
                label: 'Parent ID',
                type: 'string',
                name: 'parent_id'
              }
            ]
          },
          {
            name: 'files',
            type: 'array',
            of: 'object',
            label: 'Files',
            properties: [
              {
                control_type: 'text',
                label: 'Checksum',
                type: 'string',
                name: 'checksum'
              },
              {
                control_type: 'number',
                label: 'Size',
                parse_output: 'float_conversion',
                type: 'number',
                name: 'size'
              },
              {
                control_type: 'text',
                label: 'Path',
                type: 'string',
                name: 'path'
              },
              {
                control_type: 'text',
                label: 'Name',
                type: 'string',
                name: 'name'
              },
              {
                control_type: 'checkbox',
                label: 'Locked',
                render_input: {},
                parse_output: {},
                toggle_hint: 'Select from option list',
                toggle_field: {
                  label: 'Locked',
                  control_type: 'text',
                  toggle_hint: 'Use custom value',
                  type: 'boolean',
                  name: 'locked'
                },
                type: 'boolean',
                name: 'locked'
              },
              {
                control_type: 'checkbox',
                label: 'Is folder',
                render_input: {},
                parse_output: {},
                toggle_hint: 'Select from option list',
                toggle_field: {
                  label: 'Is folder',
                  control_type: 'text',
                  toggle_hint: 'Use custom value',
                  type: 'boolean',
                  name: 'is_folder'
                },
                type: 'boolean',
                name: 'is_folder'
              },
              {
                control_type: 'text',
                label: 'Entry ID',
                type: 'string',
                name: 'entry_id'
              },
              {
                control_type: 'text',
                label: 'Group ID',
                type: 'string',
                name: 'group_id'
              },
              {
                control_type: 'text',
                label: 'Parent ID',
                type: 'string',
                name: 'parent_id'
              },
              {
                control_type: 'date_time',
                label: 'Last modified',
                render_input: 'date_time_conversion',
                parse_output: 'date_time_conversion',
                type: 'date_time',
                name: 'last_modified'
              },
              {
                control_type: 'text',
                label: 'Uploaded by',
                type: 'string',
                name: 'uploaded_by'
              },
              {
                control_type: 'number',
                label: 'Uploaded',
                parse_output: 'float_conversion',
                type: 'number',
                name: 'uploaded'
              },
              {
                control_type: 'number',
                label: 'Num versions',
                parse_output: 'float_conversion',
                type: 'number',
                name: 'num_versions'
              }
            ]
          }
        ]
      end
    },

    event: {
      fields: lambda do |_connection, _config_fields|
        [
          { name: 'id', type: 'integer', label: 'Event ID' },
          { name: 'timestamp', type: 'date_time', label: 'Event time' },
          { name: 'actor', type: 'integer' },
          { name: 'type', label: 'Event type' },
          { name: 'action' },
          { name: 'data', type: 'object', properties: [
            { name: 'target_path' },
            { name: 'target_id' },
            { name: 'target_group_id' },
            { name: 'is_folder' }
          ] },
          { name: 'action_source', label: 'Event source' },
          { name: 'object_detail', label: 'Object URL' }
        ]
      end
    }
  },

  actions: {
    # Custom action for Egnyte
    custom_action: {
      description: "Custom <span class='provider'>action</span> " \
        "in <span class='provider'>Egnyte</span>",

      help: {
        body: 'Build your own Egnyte action with an HTTP request. The ' \
          'request will be authorized with your Egnyte connection.',
        learn_more_url: 'https://developers.egnyte.com/docs/read',
        learn_more_text: 'Egnyte API Documentation'
      },

      execute: lambda do |_connection, input|
        verb = input['verb']
        data = input.dig('input', 'data').presence || {}

        case verb
        when 'get'
          response =
            get(input['path'], data)
            .after_error_response(/.*/) do |_code, body, _header, message|
              error("#{message}: #{body}")
            end.compact

          if response.is_a?(Array)
            array_name = parse_json(input['output'] || '[]')
                         .dig(0, 'name') || 'array'
            { array_name.to_s => response }
          elsif response.is_a?(Hash)
            response
          else
            error('API response is not a JSON')
          end
        when 'post'
          post(input['path'], data)
            .after_error_response(/.*/) do |_code, body, _header, message|
            error("#{message}: #{body}")
          end.compact
        when 'put'
          put(input['path'], data)
            .after_error_response(/.*/) do |_code, body, _header, message|
            error("#{message}: #{body}")
          end.compact
        when 'patch'
          patch(input['path'], data)
            .after_error_response(/.*/) do |_code, body, _header, message|
            error("#{message}: #{body}")
          end.compact
        when 'delete'
          delete(input['path'], data)
            .after_error_response(/.*/) do |_code, body, _header, message|
            error("#{message}: #{body}")
          end.compact
        end
      end,

      config_fields: [{
        name: 'verb',
        label: 'Request type',
        hint: 'Select HTTP method of the request',
        optional: false,
        control_type: 'select',
        pick_list: %w[get post put patch delete]
          .map { |verb| [verb.upcase, verb] }
      }],

      input_fields: lambda do |object_definitions|
        object_definitions['custom_action_input']
      end,

      output_fields: lambda do |object_definitions|
        object_definitions['custom_action_output']
      end
    },

    get_file_or_folder_info: {
      title: 'Get file/folder information',
      description: "Get <span class='provider'>file or folder</span> " \
        "in <span class='provider'>Egnyte</span>",
      help: 'List information about a file or folder (including ' \
      'folder contents).',

      input_fields: lambda do |_object_definitions|
        [
          {
            name: 'path',
            hint: 'Complete path of file/folder. For e.g.: /Private/joe/doc',
            optional: false
          },
          {
            name: 'list_content',
            hint: 'For folders, this option determines whether the contents ' \
            'of the folder (files and subfolders) are included in the ' \
            'response. For files, when `Yes`, this will return a list of all ' \
            'of the versions of the file.',
            control_type: 'checkbox',
            type: 'boolean'
          },
          {
            name: 'allowed_link_types',
            hint: 'If `Yes`, then show allowed_file_link_types, ' \
            'allowed_folder_link_types fields, and allow_upload_links fields.',
            control_type: 'checkbox',
            type: 'boolean'
          },
          {
            name: 'perms',
            label: 'Permissions',
            hint: 'If `Yes`, the response includes a permissions key for ' \
            'the folder that lists the different users and groups who have ' \
            'permissions along with their permission level',
            control_type: 'checkbox',
            type: 'boolean'
          },
          {
            name: 'include_perm',
            label: 'Include permission level',
            hint: "If `Yes`, the response returns the current user's ' \
            'permission level on the folder and subfolders",
            control_type: 'checkbox',
            type: 'boolean'
          }
        ]
      end,

      execute: lambda do |_connection, input|
        element_path = input.delete('path').encode_url.gsub(/%2F/, '/')
        get("/pubapi/v1/fs/#{element_path}")
      end,

      output_fields: lambda do |object_definitions|
        object_definitions['info']
      end

      # TODO: add sample_output for all acitons and triggers
    },

    upload_file: {
      description: "Upload <span class='provider'>file</span> in " \
        "<span class='provider'>Egnyte</span>",

      input_fields: lambda { |_object_definitions|
        [
          {
            hint: 'Complete path of the file that you want to ' \
              'upload. E.g.: /Shared/Documents/Contacts.csv',
            name: 'file_path',
            optional: false
          },
          { name: 'file_content', optional: false } # ,
          # {
          #   hint: 'SHA512 hash of entire file that can be used for ' \
          #   'validating upload integrity.',
          #   name: 'X-Sha512-Checksum',
          #   optional: true,
          #   type: 'string'
          # },
          # {
          #   hint: 'Indicates last modified date for file. If omitted, the ' \
          #   'current time will be used as the last modified date.',
          #   name: 'Last-Modified',
          #   optional: true,
          #   type: 'date_time'
          # }
        ]
      },

      execute: lambda do |_connection, input|
        file_path = input['file_path'].encode_url.gsub(/%2F/, '/')

        post("/pubapi/v1/fs-content/#{file_path}")
          .request_body(input['file_content'])
      end,

      output_fields: lambda do |_object_definitions|
        [
          { name: 'checksum' },
          { name: 'group_id' },
          { name: 'entry_id' }
        ]
      end
    },

    search_file_or_folder: {
      title: 'Search file/folder',
      description: "Search <span class='provider'>file or folder</span> " \
        "in <span class='provider'>Egnyte</span>",
      help: 'Fetches the records that match the search criteria. Returns ' \
        'a maximum of 20 records.',

      input_fields: lambda { |_object_definitions|
        [
          { name: 'query', optional: false },
          {
            name: 'type',
            control_type: 'select',
            pick_list: [%w[Folder FOLDER], %w[File FILE]],
            optional: false
          }
        ]
      },

      execute: lambda { |_connection, input|
        get('/pubapi/v1/search', input)
      },

      output_fields: lambda { |_object_definitions|
        [
          {
            name: 'results',
            type: 'array',
            of: 'object',
            label: 'Results',
            properties: [
              {
                control_type: 'text',
                label: 'Name',
                type: 'string',
                name: 'name'
              },
              {
                control_type: 'text',
                label: 'Path',
                type: 'string',
                name: 'path'
              },
              {
                control_type: 'text',
                label: 'Type',
                type: 'string',
                name: 'type'
              },
              {
                control_type: 'number',
                label: 'Size',
                parse_output: 'float_conversion',
                type: 'number',
                name: 'size'
              },
              {
                control_type: 'text',
                label: 'Snippet',
                type: 'string',
                name: 'snippet'
              },
              {
                control_type: 'text',
                label: 'Entry ID',
                type: 'string',
                name: 'entry_id'
              },
              {
                control_type: 'text',
                label: 'Group ID',
                type: 'string',
                name: 'group_id'
              },
              {
                control_type: 'date_time',
                label: 'Last modified',
                render_input: 'date_time_conversion',
                parse_output: 'date_time_conversion',
                type: 'date_time',
                name: 'last_modified'
              },
              {
                control_type: 'text',
                label: 'Uploaded by',
                type: 'string',
                name: 'uploaded_by'
              },
              {
                control_type: 'text',
                label: 'Uploaded by username',
                type: 'string',
                name: 'uploaded_by_username'
              },
              {
                control_type: 'number',
                label: 'Num versions',
                parse_output: 'float_conversion',
                type: 'number',
                name: 'num_versions'
              },
              {
                control_type: 'text',
                label: 'Snippet html',
                type: 'string',
                name: 'snippet_html'
              },
              {
                control_type: 'checkbox',
                label: 'Is folder',
                toggle_hint: 'Select from option list',
                toggle_field: {
                  label: 'Is folder',
                  control_type: 'text',
                  toggle_hint: 'Use custom value',
                  type: 'boolean',
                  name: 'is_folder'
                },
                type: 'boolean',
                name: 'is_folder'
              }
            ]
          }
        ]
      }
    },

    download_file: {
      description: "Download <span class='provider'>file</span> in " \
        "<span class='provider'>Egnyte</span>",

      input_fields: lambda do |_object_definitions|
        [{
          name: 'file_path',
          optional: false,
          hint: 'Complete path of the file that you want to ' \
            'download. E.g.: /Shared/Documents/Contacts.csv'
        }]
      end,

      execute: lambda do |_connection, input|
        file_path = input['file_path'].encode_url.gsub(/%2F/, '/')

        {
          "content": get("/pubapi/v1/fs-content/#{file_path}")
            .headers('Accept' => '*/*')
            .after_error_response(/.*/) do |_code, body, _header, message|
              error("#{message}: #{body}")
            end
            .response_format_raw
            .encode('ASCII-8BIT')

        }
      end,

      output_fields: ->(_object_definitions) { [{ name: 'content' }] },

      sample_output: lambda do |_connection, _input|
        { "content": 'Hello World!' }
      end
    },

    # TODO: Need to get the folder structure dynamically based on the selection,
    # pick llist of tree type
    # control_type: 'select', pick_list: 'folders',
    create_folder: {
      description: "Create <span class='provider'>folder</span> " \
        "in <span class='provider'>Egnyte</span>",

      input_fields: lambda do |_object_definitions|
        [
          {
            name: 'parent_folder',
            optional: false,
            hint: 'Complete path of the parent folder, where you want to ' \
            'create and place this new folder. E.g.: /Shared/Documents'
          },
          { name: 'folder_name', optional: false }
        ]
      end,

      execute: lambda do |_connection, input|
        parent_folder = input['parent_folder'].encode_url.gsub(/%2F/, '/')
        folder_name = input['folder_name'].encode_url.gsub(/%2F/, '/')
        post("/pubapi/v1/fs/#{parent_folder}/#{folder_name}",
             action: 'add_folder')
      end,

      output_fields: lambda { |_object_definitions|
        [{ name: 'path' }, { name: 'folder_id' }]
      }
    },

    copy_file_or_folder: {
      title: 'Copy file or folder',
      description: "Copy <span class='provider'>file</span> or " \
      "<span class='provider'>folder</span> in <span class='provider'>Egnyte" \
      '</span>',

      input_fields: lambda do |_object_definitions|
        [
          {
            name: 'source_path',
            optional: false,
            hint: 'Complete path of the source file/folder. ' \
            'E.g.: /Shared/Documents/file.txt'
          },
          {
            name: 'destination_path',
            optional: false,
            hint: 'Complete path of the destination file/folder, where you ' \
            'want to move the file/folder. E.g.: /Shared/Documents/file.txt'
          },
          {
            name: 'permissions',
            hint: 'Determines how permissions of copied file or folder are ' \
            'derived. Allowed values are: keep_original, inherit_from_parent.' \
            ' If neither is specified, the default behavior is to use the ' \
            'workgroup settings configuration, found at Configuration -> ' \
            'Security & Authentication -> Keep permissions on folder move ' \
            '& copy',
            control_type: 'select',
            pick_list: [%w[Keep\ original keep_original],
                        %w[Inherit\ from\ parent inherit_from_parent]]
          },
          {
            name: 'entry_id',
            hint: 'Specifies the entry ID of the file version to copy.Entry ' \
            'ID for a specific version of the desired file. If none is ' \
            'specified then the latest file version for the file ' \
            'will be copied.'
          }
        ]
      end,

      execute: lambda do |_connection, input|
        source_path = input['source_path'].encode_url.gsub(/%2F/, '/')
        destination_path = input['destination_path'].encode_url.gsub(/%2F/, '/')
        post("/pubapi/v1/fs/#{source_path}",
             {
               action: 'copy',
               destination: destination_path,
               permissions: input['permissions'],
               entry_id: input['entry_id']
             }.compact)
      end,

      output_fields: lambda do |_object_definitions|
        [{ name: 'path' }, { name: 'group_id' }]
      end
    },

    move_file_or_folder: {
      title: 'Move/rename file or folder',
      description: "Move/rename <span class='provider'>file</span> or " \
      "<span class='provider'>folder</span> in <span class='provider'>Egnyte" \
      '</span>',

      input_fields: lambda do |_object_definitions|
        [
          {
            name: 'source_path',
            optional: false,
            hint: 'Complete path of the source file/folder. ' \
            'E.g.: /Shared/Documents/file.txt'
          },
          {
            name: 'destination_path',
            optional: false,
            hint: 'Complete path of the destination file/folder, where you ' \
            'want to move the file/folder. E.g.: /Shared/Documents/file.txt'
          },
          {
            name: 'permissions',
            hint: 'Determines how permissions of moved file or folder are ' \
            'derived',
            control_type: 'select',
            pick_list: [%w[Keep\ original keep_original],
                        %w[Inherit\ from\ parent inherit_from_parent]]
          }
        ]
      end,

      execute: lambda do |_connection, input|
        source_path = input['source_path'].encode_url.gsub(/%2F/, '/')
        destination_path = input['destination_path'].encode_url.gsub(/%2F/, '/')
        post("/pubapi/v1/fs/#{source_path}",
             {
               action: 'move',
               destination: destination_path,
               permissions: input['permissions']
             }.compact)
      end,

      output_fields: lambda do |_object_definitions|
        [{ name: 'path' }, { name: 'group_id' }]
      end
    },

    get_event_details: {
      title: 'Get event details',
      description: "Get <span class='provider'>file system</span> or " \
      "<span class='provider'>note</span> event details in <span class='provider'>" \
      'Egnyte</span>',
      input_fields: lambda do |_object_definitions|
        [{ name: 'object_detail', label: 'Event object URI',
           optional: false,
           hint: 'The object uri from the Event trigger output e.g ' \
           'https://appworkatodev.qa-egnyte.com/pubapi/v1/fs/Shared/Bennett%20Test' }]
      end,
      execute: lambda do |_conneciton, input|
        get(input['object_detail'])
      end,
      output_fields: lambda do |object_definitions|
        object_definitions['file']
      end,
      sample_output: lambda do |_connection, input|
      end
    }
  },

  triggers: {
    # Event triggered when a folder is added/updated in the given directory
    # TO DO: pick list to show the folder tree strcture
    new_or_updated_file: {
      title: 'New/updated file in directory',
      description: "New/updated <span class='provider'>file</span> " \
        "in <span class='provider'>Egnyte</span>",

      input_fields: lambda do |_object_definitions|
        [
          { name: 'folder_path', label: 'Folder', control_type: 'tree',
            tree_options: { selectable_folder: true },
            pick_list: :folders,
            sticky: true,
            hint: 'Scope requests to a given folder. When folder is' \
            ' specified, it return events that have occurred at this path ' \
            '(including events inside subfolders).',
            toggle_hint: 'Select folder',
            toggle_field: {
              name: 'folder_path',
              type: 'string',
              control_type: 'text',
              label: 'Folder path',
              sticky: true,
              toggle_hint: 'Use folder Path',
              hint: 'Folder relative path. the folder parameter scope requests to a given folder. When folder is' \
              ' specified, it return events that have occurred at this path ' \
              '(including events inside subfolders).'
            } }
        ]
      end,

      poll: lambda do |_connection, input, offset|
        offset ||= 0
        # TODO: fix this
        page_size = 100
        # page_size = 5
        folder_path = input['folder_path'].encode_url.gsub(/%2F/, '/')
        response = get("/pubapi/v1/fs#{folder_path}",
                       count: page_size,
                       offset: offset,
                       sort_by: 'last_modified',
                       sort_direction: 'descending')
        next_offset = offset + page_size
        can_poll_more = (response['total_count'] >= next_offset)

        {
          events: response['files'] || [],
          next_poll: can_poll_more ? next_offset : 0,
          can_poll_more: can_poll_more
        }
      end,

      dedup: ->(file) { "#{file['group_id']}@#{file['last_modified']}" },

      output_fields: ->(object_definitions) { object_definitions['file'] }

      # TODO: add sample_output
    },

    new_events: {
      title: 'New/updated/deleted events',
      description: "New/updated/deleted <span class='provider'>events</span> " \
        "in <span class='provider'>Egnyte</span>",
      help: {
        body: 'The Events API lets your application learn about activity in Egnyte.' \
        ' This means you can find out what has happened since a given moment in time, ' \
        'letting your application respond or react to activity. Currently, this API is' \
        ' in beta and is limited to file system and notes activity.<br>By default, ' \
        'the trigger retrieves events from the Recipe start time.' \
        ' <b>Event ID</b> takes' \
        ' priority when <b>Event ID</b> and <b>Retrieve events from beginning</b> options specified.',
        learn_more_text: 'Click here to learn more on Events API ',
        learn_more_url: 'https://developers.egnyte.com/docs/read/Events_API'
      },
      config_fields: [
        {
          name: 'event_types',
          optional: false,
          control_type: 'multiselect',
          pick_list: 'event_types',
          pick_list_params: {},
          delimiter: ',',
          toggle_hint: 'Select acecess scopes',
          toggle_field: {
            name: 'event_types',
            label: 'Event types',
            type: 'string',
            control_type: 'text',
            toggle_hint: 'Proivde custom values',
            hint: 'Allowed values: one ore more  of (file_system,note) separated by comma '
          },
          hint: 'Select the event types you want to be notified'
        }
      ],
      input_fields: lambda do |_object_definitions|
        [
          { name: 'folder', label: 'Folder', control_type: 'tree',
            tree_options: { selectable_folder: true },
            pick_list: :folders,
            sticky: true,
            hint: 'Scope requests to a given folder. When folder is' \
            ' specified, it return events that have occurred at this path ' \
            '(including events inside subfolders).',
            toggle_hint: 'Select folder',
            toggle_field: {
              name: 'folder',
              type: 'string',
              control_type: 'text',
              label: 'Folder path',
              sticky: true,
              toggle_hint: 'Use folder Path',
              hint: 'Folder relative path. the folder parameter scope requests to a given folder. When folder is' \
              ' specified, it return events that have occurred at this path ' \
              '(including events inside subfolders).'
            } },
          {
            name: 'event_id',
            type: 'integer',
            sticky: true,
            hint: 'Event ID to fetch from, if Event ID is not provide, trigger ' \
            'fetches from the beginning e.g. 51'
          },
          {
            name: 'event_from',
            label: 'Retrieve events from beginning',
            type: 'boolean',
            control_type: 'checkbox',
            sticky: true,
            hint: 'Select Yes, to fetch events from the beginning of time. otherwise, it retrieves events from ' \
              'Recipe start time if <b>No</b> or unselected'
          }
        ]
      end,
      poll: lambda do |_connection, input, closure|
        page_size = 100
        event_types =
          closure&.[]('event_types') || input['event_types']&.gsub(',', '|')
        # get the cursor_value, only first time, cursor endpoint is called
        latest_event_id =
          closure&.[]('latest_event_id') || input['event_id'] ||
          if input['event_from'] == true
            get('/pubapi/v1/events/cursor')&.dig('oldest_event_id')
          else
            get('/pubapi/v1/events/cursor')&.dig('latest_event_id')
          end
        folder = closure&.[]('folder') || input['folder'].gsub('%20', ' ').gsub(/%2F/, '/')

        events = get('/pubapi/v1/events')
                 .params(id: latest_event_id,
                         type: event_types,
                         folder: folder,
                         count: page_size)
                 .after_response do |code, body, _headers|
                   if code == 204
                     []
                   else
                     body['events']
                   end
                 end
        closure = if events&.size > 0
                    { 'event_types' => event_types,
                      'folder' => folder,
                      'latest_event_id' => events.last['id'] }
                  else
                    { 'event_types' => event_types,
                      'folder' => folder,
                      'latest_event_id' => latest_event_id }
                  end
        {
          events: events,
          next_poll: closure,
          can_poll_more: events.size >= page_size
        }
      end,
      dedup: lambda do |event|
        "#{event['id']}__#{event['type']}__#{event['timestamp']}"
      end,
      output_fields: lambda do |object_definitions|
        object_definitions['event']
      end,
      sample_output: lambda do |_connection, _input|
        call('event_sample_output')
      end
    }
  },

  pick_lists: {
    event_types: lambda do |_connection|
      [['File system', 'file_system'], %w[Notes note]]
    end,
    folders: lambda do |_connection, **args|
      # Get sub folders
      if (folder_path = args&.[](:__parent_id)).presence
        get("/pubapi/v1/fs/#{folder_path}").
          params(list_content: true,
                 sort_by: 'name')['folders']&.map do |folder|
          [folder['name'].labelize, folder['path'].gsub(' ', '%20'), folder['path'], true]
        end
      else
        # Get root folders
        get('/pubapi/v1/fs/')['folders']&.
          map do |folder|
            [folder['name'].labelize, folder['path'].gsub(' ', '%20'), folder['path'], true]
          end
      end
    end,
    suppress_list: lambda do |_connection|
      [%w[Application app], %w[User user]]
    end
  }
}
